#!/usr/bin/env python3
import os, re, subprocess, sys, argparse
from pathlib import Path

# Defaults (you can override via CLI)
DEFAULT_FRAMERATE = 10      # Set to 1 if your AE comps exported at 1 fps
DEFAULT_CRf       = 18      # 0-51; lower = higher quality
DEFAULT_PRESET    = "veryfast"  # ultrafast … placebo

def detect_sequence_pattern(png_dir: Path):
    """
    Find zero-padded PNG sequence like 0000.png or 00000.png and return ('%04d', 'png')
    Returns (None, None) if no sequence is detected.
    """
    files = sorted([p.name for p in png_dir.iterdir() if p.is_file() and p.suffix.lower()==".png"])
    if not files:
        return None, None

    # Look for names like 0000.png, 0001.png, etc.
    m = None
    for fn in files:
        m = re.match(r'^(\d+)\.png$', fn, re.IGNORECASE)
        if m:
            pad = len(m.group(1))
            return f"%0{pad}d", "png"

    # If not strictly numeric, look for a prefix + zero-padded number e.g. name_0000.png
    # ffmpeg needs the INPUT as something like name_%04d.png
    # Try to find a common (prefix, pad) among files.
    candidates = []
    for fn in files:
        m = re.match(r'^(.*?)(\d+)\.png$', fn, re.IGNORECASE)
        if m:
            prefix, nums = m.group(1), m.group(2)
            candidates.append((prefix, len(nums)))
    if not candidates:
        return None, None

    # Pick the most common (prefix, pad)
    from collections import Counter
    (prefix, pad), _count = Counter(candidates).most_common(1)[0]
    return f"{prefix}%0{pad}d", "png"

def build_ffmpeg_cmd(input_dir: Path, pattern: str, ext: str, out_file: Path,
                     framerate: int, crf: int, preset: str):
    """
    pattern is either like '%04d' (pure numeric) or 'name_%04d'
    Final input passed to ffmpeg is '<pattern>.<ext>'
    """
    input_pattern = f"{pattern}.{ext}"

    # We set input framerate with -framerate (applies to image2 demuxer).
    # yuv420p maximizes compatibility. No scaling applied, so pixel art stays crisp.
    # If you want mathematically lossless, switch codec to libx264rgb + -pix_fmt rgb24 (but larger files).
    cmd = [
        "ffmpeg",
        "-y",
        "-hide_banner",
        "-loglevel", "error",
        "-framerate", str(framerate),
        "-i", str(input_dir / input_pattern),
        "-c:v", "libx264",
        "-pix_fmt", "yuv420p",
        "-crf", str(crf),
        "-preset", preset,
        "-r", str(framerate),           # output fps
        str(out_file)
    ]
    return cmd

def make_mp4s_for_league(league_dir: Path, framerate: int, crf: int, preset: str, dry_run: bool=False):
    if not league_dir.is_dir():
        raise SystemExit(f"Not a directory: {league_dir}")

    out_root = league_dir / "MP4s"
    out_root.mkdir(parents=True, exist_ok=True)

    # Iterate immediate subfolders (each expected to contain a PNG sequence)
    subdirs = [p for p in league_dir.iterdir() if p.is_dir() and p.name != "MP4s"]
    if not subdirs:
        print("No subfolders found to process.")
        return

    print(f"Found {len(subdirs)} subfolder(s) in {league_dir}")
    for seq_dir in sorted(subdirs):
        pattern, ext = detect_sequence_pattern(seq_dir)
        if not pattern:
            print(f"⚠️  No PNG sequence detected in: {seq_dir.name} — skipping.")
            continue

        out_file = out_root / f"{seq_dir.name}.mp4"
        cmd = build_ffmpeg_cmd(seq_dir, pattern, ext, out_file, framerate, crf, preset)

        print(f"→ {seq_dir.name}  [{pattern}.{ext}]  →  {out_file.name}")
        if dry_run:
            print("   DRY-RUN:", " ".join(map(str, cmd)))
            continue

        try:
            subprocess.run(cmd, check=True)
        except subprocess.CalledProcessError as e:
            print(f"❌ ffmpeg failed for {seq_dir}: {e}")

def main():
    print("Enter the path to your league directory (e.g., /Users/.../FG_Animation_1_384/NFL):")
    league_dir = input("> ").strip()

    if not league_dir:
        print("❌ No directory provided, exiting.")
        return

    fps    = DEFAULT_FRAMERATE
    crf    = DEFAULT_CRf
    preset = DEFAULT_PRESET

    make_mp4s_for_league(Path(league_dir), fps, crf, preset, dry_run=False)
    print("✅ Done.")


if __name__ == "__main__":
    main()
